<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class OSF_Custom_Post_Type_Case_Study
 */
class OSF_Custom_Post_Type_Portfolio extends OSF_Custom_Post_Type_Abstract {
    public $post_type = 'osf_portfolio';
    public $taxonomy = 'osf_portfolio_cat';

    static $instance;

    public static function getInstance() {
        if (!isset(self::$instance) && !(self::$instance instanceof OSF_Custom_Post_Type_Portfolio)) {
            self::$instance = new OSF_Custom_Post_Type_Portfolio();
        }

        return self::$instance;
    }

    /**
     * @return void
     */
    public function create_post_type() {

        $labels = array(
            'name'               => __('Portfolios', 'startor-core'),
            'singular_name'      => __('Portfolio', 'startor-core'),
            'add_new'            => __('Add New Portfolio', 'startor-core'),
            'add_new_item'       => __('Add New Portfolio', 'startor-core'),
            'edit_item'          => __('Edit Portfolio', 'startor-core'),
            'new_item'           => __('New Portfolio', 'startor-core'),
            'view_item'          => __('View Portfolio', 'startor-core'),
            'search_items'       => __('Search Portfolios', 'startor-core'),
            'not_found'          => __('No Portfolios found', 'startor-core'),
            'not_found_in_trash' => __('No Portfolios found in Trash', 'startor-core'),
            'parent_item_colon'  => __('Parent Portfolio:', 'startor-core'),
            'menu_name'          => __('Portfolios', 'startor-core'),
        );

        $labels = apply_filters('osf_postype_portfolio_labels', $labels);
        $slug_field = osf_get_option('portfolio_settings', 'slug_portfolio', 'portfolio');

        register_post_type($this->post_type,
            array(
                'labels'        => $labels,
                'supports'      => array('title', 'editor', 'excerpt', 'thumbnail'),
                'public'        => true,
                'has_archive'   => true,
                'rewrite'       => array('slug' => $slug_field),
                'menu_position' => 5,
                'categories'    => array(),
                'menu_icon'     => $this->get_icon(__FILE__),
                'taxonomies'    => array('post_tag')
            )
        );
    }

    /**
     * @return void
     */
    public function create_taxonomy() {
        $labels = array(
            'name'              => __('Categories', 'startor-core'),
            'singular_name'     => __('Category', 'startor-core'),
            'search_items'      => __('Search Category', 'startor-core'),
            'all_items'         => __('All Categories', 'startor-core'),
            'parent_item'       => __('Parent Category', 'startor-core'),
            'parent_item_colon' => __('Parent Category:', 'startor-core'),
            'edit_item'         => __('Edit Category', 'startor-core'),
            'update_item'       => __('Update Category', 'startor-core'),
            'add_new_item'      => __('Add New Category', 'startor-core'),
            'new_item_name'     => __('New Category Name', 'startor-core'),
            'menu_name'         => __('Categories', 'startor-core'),
        );
        $labels = apply_filters('osf_postype_portfolio_cat_labels', $labels);
        $slug_cat_field = osf_get_option('portfolio_settings', 'slug_category_portfolio', 'category-portfolio');
        $args = array(
            'hierarchical'      => true,
            'labels'            => $labels,
            'show_ui'           => true,
            'show_admin_column' => true,
            'query_var'         => true,
            'show_in_nav_menus' => true,
            'rewrite'           => array('slug' => $slug_cat_field)
        );
        // Now register the taxonomy
        register_taxonomy($this->taxonomy, array($this->post_type), $args);
    }

    /**
     * @param $classes
     *
     * @return array
     */
    public function body_class($classes) {
        if (is_post_type_archive($this->post_type) || is_tax($this->taxonomy)) {
            $classes[] = 'opal-content-layout-' . get_theme_mod('osf_portfolio_archive_layout', '1c');
        } else if (is_singular($this->post_type)) {
            $classes[] = 'opal-content-layout-' . get_theme_mod('osf_portfolio_single_layout', '1c');
        }
        return $classes;
    }

    /**
     * @param array $arg
     *
     * @return WP_Query
     */
    public function create_query($per_page = -1, $taxonomies = array()) {
        $args = array(
            'post_type'      => $this->post_type,
            'posts_per_page' => $per_page,
            'post_status'    => 'publish',
        );
        if (!empty($taxonomies)) {
            $args ['tax_query'] = array(
                'taxonomy' => $this->taxonomy,
                'field'    => 'slug',
                'terms'    => $taxonomies
            );
        }
        return new WP_Query($args);
    }

    public function create_meta_box() {
        $prefix = 'osf_';

        $cmb2 = new_cmb2_box(array(
            'id'           => $prefix . 'portfolio_setting',
            'title'        => __('Portfolio Infomation', 'startor-core'),
            'object_types' => array('osf_portfolio'),
        ));

	    $group_field_id = $cmb2->add_field( array(
		    'id'          => 'osf_portfolio_repeat_group',
		    'type'        => 'group',
		    'options'     => array(
			    'group_title'       => __( 'Entry {#}', 'startor-core' ), // since version 1.1.4, {#} gets replaced by row number
			    'add_button'        => __( 'Add Another Entry', 'startor-core' ),
			    'remove_button'     => __( 'Remove Entry', 'startor-core' ),
			    'sortable'          => true,

		    ),
	    ) );

	    $cmb2->add_group_field( $group_field_id, array(
		    'name' => 'Entry Title',
		    'id'   => 'title',
		    'type' => 'text',
	    ) );

	    $cmb2->add_group_field( $group_field_id, array(
		    'name' => 'Description',
		    'id'   => 'description',
		    'type' => 'textarea_small',
	    ) );

        $cmb2->add_field(array(
            'name' => 'Gallery',
            'desc' => '',
            'id'   => $prefix . 'portfolio_gallery',
            'type' => 'file_list',
        ));

    }



    /**
     * @return array|int|WP_Error
     */
    public function get_terms() {
        return get_terms(array($this->taxonomy));
    }

    public function get_term_portfolio($post_id) {
        $terms = get_the_terms($post_id, $this->taxonomy);
        $output = '';
        if (!is_wp_error($terms) && is_array($terms)) {
            foreach ($terms as $key => $term) {
                $term_link = get_term_link($term);
                if (is_wp_error($term_link)) {
                    continue;
                }
                $output .= '<a href="' . esc_url($term_link) . '">' . $term->name . '</a>';
                if ($key < count($terms) - 1) {
                    $output .= ', ';
                }
            }

        }
        return $output;
    }

    function startor_fnc_related_portfolio($relate_count = 3, $posttype = 'osf_portfolio', $taxonomy = 'osf_portfolio_cat') {

        $terms = get_the_terms(get_the_ID(), $taxonomy);
        $termids = array();

        if ($terms) {
            foreach ($terms as $term) {
                $termids[] = $term->term_id;
            }
        }

        $args = array(
            'post_type'      => $posttype,
            'posts_per_page' => $relate_count,
            'post__not_in'   => array(get_the_ID()),
            'tax_query'      => array(
                'relation' => 'AND',
                array(
                    'taxonomy' => $taxonomy,
                    'field'    => 'id',
                    'terms'    => $termids,
                    'operator' => 'IN'
                )
            )
        );

        $related = new WP_Query($args);

        if ($related->have_posts()) {
            echo '<div class="related-portfolio">';
            echo '<h3 class="related-heading">' . esc_html__('Related Project', 'startor-core') . '</h3>';
            echo '<div class="row elementor-portfolio-style-default" data-elementor-columns="3">';
            while ($related->have_posts()) : $related->the_post();
                ?>
                <div class="column-item portfolio-entries">
                    <?php get_template_part('template-parts/portfolio/content'); ?>
                </div>
            <?php
            endwhile;
            echo '</div>';
            echo '</div>';

            wp_reset_postdata();
        }


    }

    /**
     * @param $wp_customize WP_Customize_Manager
     */
    public function customize_register($wp_customize) {

        $wp_customize->add_panel('osf_portfolio', array(
            'title'      => __('Portfolio', 'startor-core'),
            'capability' => 'edit_theme_options',
            'priority'   => 1,
        ));

        //Portfolio Archive config
        $wp_customize->add_section('osf_portfolio_archive', array(
            'title'      => __('Archive', 'startor-core'),
            'capability' => 'edit_theme_options',
            'panel'      => 'osf_portfolio',
            'priority'   => 1,
        ));

        // =========================================
        // Select Layout
        // =========================================

        $wp_customize->add_control(new OSF_Customize_Control_Image_Select($wp_customize, 'osf_portfolio_archive_layout', array(
            'section' => 'osf_portfolio_archive',
            'label'   => __('Select Layout', 'startor-core'),
            'choices' => $this->options,
        )));

        $wp_customize->add_setting('osf_portfolio_archive_column', array(
            'default'           => '3',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        $wp_customize->add_control('osf_portfolio_archive_column', array(
            'section' => 'osf_portfolio_archive',
            'label'   => __('Columns', 'startor-core'),
            'type'    => 'select',
            'choices' => array(
                '1' => __('1 Column', 'startor-core'),
                '2' => __('2 Columns', 'startor-core'),
                '3' => __('3 Columns', 'startor-core'),
                '4' => __('4 Columns', 'startor-core'),
            ),
        ));

        if (class_exists('OSF_Customize_Control_Button_Group')) {
            $wp_customize->add_setting('osf_portfolio_archive_style', array(
                'default'           => 'default',
                'sanitize_callback' => 'sanitize_text_field',
            ));
            $wp_customize->add_control(new OSF_Customize_Control_Button_Group($wp_customize, 'osf_portfolio_archive_style', array(
                'section' => 'osf_portfolio_archive',
                'label'   => __('Select Style', 'startor-core'),
                'default' => 'default',
                'choices' => array(
                    'default' => __('Default', 'startor-core'),
                    'overlay' => __('Overlay', 'startor-core'),
                    'caption' => __('Caption', 'startor-core'),
                ),
            )));
        }

        //Single portfolio layout
        $wp_customize->add_section('osf_portfolio_single', array(
            'title'      => __('Single', 'startor-core'),
            'capability' => 'edit_theme_options',
            'panel'      => 'osf_portfolio',
            'priority'   => 1,
        ));

        if (class_exists('OSF_Customize_Control_Button_Group')) {
            $wp_customize->add_setting('osf_portfolio_single_layout_style', array(
                'default'           => '',
                'sanitize_callback' => 'sanitize_text_field',
            ));
            $wp_customize->add_control(new OSF_Customize_Control_Button_Group($wp_customize, 'osf_portfolio_single_layout_style', array(
                'section' => 'osf_portfolio_single',
                'label'   => __('Select Layout Style', 'startor-core'),
                'choices' => array(
                    'full'        => __('Full', 'startor-core'),
                    'sidebar' => __('Sidebar', 'startor-core'),
                ),
            )));
        }

        if (class_exists('OSF_Customize_Control_Button_Group')) {
            $wp_customize->add_setting('osf_portfolio_single_gallery_style', array(
                'default'           => '',
                'sanitize_callback' => 'sanitize_text_field',
            ));
            $wp_customize->add_control(new OSF_Customize_Control_Button_Group($wp_customize, 'osf_portfolio_single_gallery_style', array(
                'section' => 'osf_portfolio_single',
                'label'   => __('Select Gallery Style', 'startor-core'),
                'choices' => array(
                    ''        => __('Gallery', 'startor-core'),
                    'gallery' => __('Carousel', 'startor-core'),
                ),
            )));
        }

        $wp_customize->add_setting('osf_portfolio_single_gallery_columns', array(
            'default'           => '3',
            'sanitize_callback' => 'sanitize_text_field',
        ));

        $wp_customize->add_control('osf_portfolio_single_gallery_columns', array(
            'section' => 'osf_portfolio_single',
            'label'   => __('Gallery Columns', 'startor-core'),
            'type'    => 'select',
            'choices' => array(
                '1' => __('1 Column', 'startor-core'),
                '2' => __('2 Columns', 'startor-core'),
                '3' => __('3 Columns', 'startor-core'),
            ),
        ));

    }

}

OSF_Custom_Post_Type_Portfolio::getInstance();

function osf_override_meta_value( $value, $object_id, $args, $field ) {
	static $defaults = null;

	// Only set the default if the original value has not been overridden
	if ( 'cmb2_field_no_override_val' !== $value ) {
		return $value;
	}

	$data = 'options-page' === $args['type']
		? cmb2_options( $args['id'] )->get( $args['field_id'] )
		: get_metadata( $args['type'], $args['id'], $args['field_id'], ( $args['single'] || $args['repeat'] ) );


	$field_id = $args['field_id'];

	// This DOES work
	if( 'osf_portfolio_repeat_group' === $field_id ) {

        // Get the default values from JSON
        if (null === $defaults) {
            // Get your JSON blob.. hard-coded for demo.
            $json = '[{"title":"Clients","description":""},{"title":"Completed","description":""},{"title":"Location","description":""},{"title":"Budgets","description":""},{"title":"Project url","description":""}]';

            $defaults = json_decode($json, 1);
        }

        $value = $defaults;


        if (!empty($data)) {
            $value = array();
            // Then loop the defaults and mash the field's value up w/ the default.
            foreach ($defaults as $key => $default_group_val) {
                $value[$key] = isset($data[$key])
                    ? wp_parse_args($data[$key], $default_group_val)
                    : $default_group_val;
            }
        }
    }

	return $value;
}

add_filter( 'cmb2_override_meta_value', 'osf_override_meta_value', 10, 4 );